package com.burhanstore.neonbubblequest.Utils.monetize

import android.app.Activity
import android.util.Log
import com.burhanstore.neonbubblequest.Utils.AppString
import com.burhanstore.neonbubblequest.Utils.appuser.UserManager
import com.vungle.warren.InitCallback
import com.vungle.warren.LoadAdCallback
import com.vungle.warren.PlayAdCallback
import com.vungle.warren.Vungle
import com.vungle.warren.error.VungleException


object Vungle_Ad {

    private const val TAG = "Vungle_Ad"

    private var placementId: String = ""
    private var isInitialized = false
    private var isLoading = false
    private lateinit var activity: Activity

    private var onAdCompleteCallback: (() -> Unit)? = null
    private var onAdFailedCallback: (() -> Unit)? = null
    private var onAdClickedCallback: (() -> Unit)? = null

    fun initialize(
        activity: Activity,
        onInitialized: (() -> Unit)? = null
    ) {
        if (isInitialized) {
            Log.d(TAG, "⚠️ Already initialized")
            return
        }

        this.activity = activity

        // Read ad unit ID from JSON settings
        val jsonReader =
            AppString.JsonReader(UserManager.getInstance(activity).getSetting()?.ad_unit ?: "")
        val vuAppId = jsonReader.get("vungle_appid")
        placementId = jsonReader.get("vungle_rewardid")

        Log.e("app run id", "vuAppId" + vuAppId + "v_rewardid " + placementId)

        if (vuAppId.isNullOrEmpty() || placementId.isEmpty()) {
            Log.e(TAG, "❌ Vungle IDs missing in config")
            return
        }

        Vungle.init(vuAppId, activity.applicationContext, object : InitCallback {
            override fun onSuccess() {
                Log.d(TAG, "✅ Vungle Initialized")
                isInitialized = true
                loadAd()
                onInitialized?.invoke()
            }

            override fun onError(exception: VungleException?) {
                Log.e(TAG, "❌ Vungle Init Error: ${exception?.localizedMessage}")
                onAdFailedCallback?.invoke()
            }

            override fun onAutoCacheAdAvailable(placementId: String) {
                Log.d(TAG, "🎯 Auto cached ad ready: $placementId")
            }
        })
    }

    /** Load a rewarded ad */
    fun loadAd() {
        if (!isInitialized) {
            Log.w(TAG, "⚠️ Vungle not initialized yet")
            return
        }
        if (isLoading) {
            Log.d(TAG, "⏳ Already loading an ad…")
            return
        }

        isLoading = true
        Vungle.loadAd(placementId, object : LoadAdCallback {
            override fun onAdLoad(id: String) {
                Log.d(TAG, "✅ Vungle Ad Loaded: $id")
                isLoading = false
            }

            override fun onError(id: String, exception: VungleException) {
                Log.e(TAG, "❌ Vungle Load Error: ${exception.localizedMessage}")
                isLoading = false
                onAdFailedCallback?.invoke()
                clearCallbacks()
            }
        })
    }

    /** Show rewarded ad */
    fun showAd(
        onAdComplete: () -> Unit = {},
        onAdFailed: (() -> Unit)? = null,
        onAdClicked: (() -> Unit)? = null
    ) {
        onAdCompleteCallback = onAdComplete
        onAdFailedCallback = onAdFailed
        onAdClickedCallback = onAdClicked

        if (Vungle.canPlayAd(placementId)) {
            Vungle.playAd(placementId, null, object : PlayAdCallback {
                override fun creativeId(creativeId: String?) {
                    Log.d(TAG, "Creative ID: $creativeId")
                }

                override fun onAdStart(id: String) {
                    Log.d(TAG, "▶️ Vungle Ad started: $id")
                }

                override fun onAdEnd(
                    placementId: String?,
                    completed: Boolean,
                    isCTAClicked: Boolean
                ) {
                    Log.d(
                        TAG,
                        "⏹ Vungle Ad ended: $placementId | completed=$completed | CTA=$isCTAClicked"
                    )
                    if (completed) {
                        onAdCompleteCallback?.invoke()
                    }
                    clearCallbacks()
                    loadAd() // Preload next ad
                }

                override fun onAdEnd(placementId: String?) {
                    Log.d(TAG, "⏹ Vungle Ad ended (no reward info): $placementId")
                }

                override fun onAdClick(id: String) {
                    Log.d(TAG, "👆 Vungle Ad clicked: $id")
                    onAdClickedCallback?.invoke()
                }

                override fun onAdRewarded(placementId: String?) {
                    Log.d(TAG, "🎁 Reward triggered for placement: $placementId")
                }

                override fun onAdLeftApplication(placementId: String?) {
                    Log.d(TAG, "↩️ User left app from Vungle Ad: $placementId")
                }

                override fun onError(placementId: String?, exception: VungleException?) {
                    Log.e(TAG, "❌ Vungle Play Error: ${exception?.localizedMessage}")
                    onAdFailedCallback?.invoke()
                    clearCallbacks()
                }

                override fun onAdViewed(placementId: String?) {
                    Log.d(TAG, "👀 Vungle Ad viewed: $placementId")
                }
            })
        } else {
            Log.w(TAG, "⚠️ Vungle Ad not ready, loading again")
            onAdFailedCallback?.invoke()
            clearCallbacks()
            loadAd()
        }
    }

    private fun clearCallbacks() {
        onAdCompleteCallback = null
        onAdFailedCallback = null
        onAdClickedCallback = null
    }
}

